function parsed_stk = stk_parse(stk,exp_info)
%A function to parse out the experimental trials into contiguous blocks and
%store them in a structure.
%Syntax:    parsed_stk = stk_parse(stk,exp_info);
%Input: stk = the image stack, which is the raw data collected.
%       exp_info = the information collected about the experiment.
%Output:    parsed_stk = It is a structure that contains in seperate
%           headings: the stimulus_stk and delay_stks

%Initialize variables
delete_frames = [];

%Read exp_info
delay_num = exp_info(1).delay;          %number of delays
trials = exp_info(1).trials;        %number of trials
exp_type = exp_info(1).exp_type;    %the type of experiment to be parsed

%Get frame number and double check for error.
total_frames = size(stk,3);         %total number of images in the stack
if total_frames ~= exp_info(1).total_frames
    error(['Error: the size of the stack does not match the number of frames reported in the experimental information.']);
end

%calculate the stimulus frames and the delay frames.
stim_frames = total_frames/(delay_num+1);   %total number of images captured accompanying a stimulus
stim_num = stim_frames/trials;          %the number of stim frames per trial
if delay_num~=0;    %there are delay frames
    %error check
    if stim_frames+delay_num*stim_frames~=total_frames
        error(['Error: the number of delay frames + data frames does not match the stack frames.']);
    end
    %extract out the delay/baseline frames, create the raw delay_stks
    for i = 1:delay_num
        delay(i).frames = stk_mod(stk,'extract','EvO',[i+1 delay_num+1]);
        delay_frames(1,i) = size(delay(i).frames,3);
        disp(['Delay ',num2str(i),' has ',num2str(delay_frames(1,i)),' frames.'])
    end
    %delete the delay frames thus creating the raw stim_stk
    for j = 1:delay_num
        delete_frames = horzcat(delete_frames,[j+1:delay_num+1:total_frames]);
    end
    stk = stk_mod(stk,'delete',delete_frames);
    %error check
    if stim_frames~=size(stk,3)
        error(['Error: Your stimulus frames does not match the calculated number.']);
    end
    %Display the size of the stim stacks for visual check.
    disp(['Stim stack has ', num2str(stim_frames),' frames.'])
    %store some delay related information.
    parsed_stk(1).delay_frames_num = delay_frames;      %the number of frames for each delay in a [x y z] vector.
    parsed_stk(1).delay_frame_index = delete_frames;    %a vector of all the delete frames in the original image stack.
end

%parse and order the raw stacks into easily processed data stacks for each
%experiment type.
switch exp_type
    case {'spatial','spatial_moving_background'}
        %This function orders the data stacks from the above functions so
        %that each stack is in sequential order in terms of the stimuli
        %number, and since the stimuli number is a sequential tile of the
        %screen so that the top left hand corner of the screen is 1 and the
        %bottom right hand corner is end, going sequentually through the
        %stack is a look at the data of the response to a stimuli in the
        %moving accross the screen from top to bottom.
        %parse some experiment specific data.
        index = exp_info(1).index;
        %error check 
        if stim_frames~=size(index,1)
            error(['Error: your index size does not match the stim frames in your stack.']);
        end
        %parse and sort the data sets: the final structure parsed_stk, will
        %contain: parsed_stk(trials).stim_stk(stim_num) and
        %parse_stk(trials).delay_stks(delays).stk(stim_num).  So, each
        %trial has its own sorted stim stack and delay stacks.
        %create a progress bar, because this might take a while, note
        %that I am using progressbar here, although waitbar is good
        %too, each has its pros and cons.  I used waitbar for
        %stack_gui, so progressbar here is just for kicks.
        prog_idx = 1;       %initiate counter for progress bar
        progressbar         %create progress bar and start timer
        for i = 1:trials
            frame_idx_trial = index(i*stim_num-stim_num+1:i*stim_num,1);    %a frame index for each trial.
            for k = 1:stim_num
                parsed_stk(i).stim_stk(:,:,frame_idx_trial(k,1)) = stk(:,:,k);
                %sort the baseline/delay frames according to the stim that preceeds it.
                if delay_num~=0     %If there are delay frames
                    for l = 1:delay_num
                        parsed_stk(i).delay_stks(l).stk(:,:,frame_idx_trial(k,1)) = delay(l).frames(:,:,k);
                        prog_idx = prog_idx+1;                  %tick the progress by a frame
                    end
                end
                %store stimulus location in an vertical array
                stim_loc(k,:) = exp_info(k).stim;
                prog_idx = prog_idx+1;                  %tick the progress by a frame
                progressbar(prog_idx/total_frames);     %update the progress
            end
            %delete the stored frames
            stk = stk_mod(stk,'delete',[1:stim_num]);
            if delay_num~=0     %If there are delay frames
                for l = 1:delay_num
                    delay(l).frames = stk_mod(delay(l).frames,'delete',[1:stim_num]);
                end
            end
            %store the frame index and the stimulus for each trial.
            parsed_stk(i).stim_loc = stim_loc;
            parsed_stk(i).index = frame_idx_trial;
        end
    case 'spatial_moving'
        %this function orders the data from the above experiment, because
        %spatial moving already has its data in sequential order in terms
        %of location on screen we are just going to sort the stacks in
        %terms of trials and direction of movement.
        %parse some experiment specific data.
        rez = exp_info(1).resolution;           %screen resolution during experiment
        stim_size = exp_info(1).stim_size;      %stimulus size
        horz_stim_num = rez(4)/stim_size;       %number of horizontal stimuli
        vert_stim_num = rez(3)/stim_size;       %number of vertical stimuli
        %error check: all calculations correct
        if (horz_stim_num*2+vert_stim_num*2)*trials~=stim_frames
            error(['Error: your image stack size for the stim frames does not match the calculated value.']);
        end
        %remember the experiment progresses: easterly(1), westerly(2), southerly(3),
        %northerly(4)...repeat...
        %parse and sort the data sets: the final structure parsed_stk, will
        %contain: parsed_stk(trials).direction(1).stk(stim_num) and
        %parse_stk(trials).delay_stks(delays).direction(1).stk(stim_num).  So, each
        %trial has its own sorted stim stack and delay stacks.
        progressbar         %create progress bar and start timer
        for i = 1:trials
            parsed_stk(i).direction(1).stim_stk = stk(:,:,1:horz_stim_num);
            parsed_stk(i).direction(2).stim_stk = stk(:,:,horz_stim_num+1:horz_stim_num*2);
            parsed_stk(i).direction(3).stim_stk = ...
                stk(:,:,horz_stim_num*2+1:horz_stim_num*2+vert_stim_num);
            parsed_stk(i).direction(4).stim_stk = ...
                stk(:,:,horz_stim_num*2+vert_stim_num+1:horz_stim_num*2+vert_stim_num*2);
            if delay_num~=0     %If there are delay frames
                for k = 1:delay_num
                    parsed_stk(i).direction(1).delay_stks(k).stk = delay(k).frames(:,:,1:horz_stim_num);
                    parsed_stk(i).direction(2).delay_stks(k).stk = delay(k).frames(:,:,horz_stim_num+1:horz_stim_num*2);
                    parsed_stk(i).direction(3).delay_stks(k).stk = ...
                        delay(k).frames(:,:,horz_stim_num*2+1:horz_stim_num*2+vert_stim_num);
                    parsed_stk(i).direction(4).delay_stks(k).stk = ...
                        delay(k).frames(:,:,horz_stim_num*2+vert_stim_num+1:horz_stim_num*2+vert_stim_num*2);
                end
            end            
            %delete the stored frames
            stk = stk_mod(stk,'delete',[1:stim_num]);
            if delay_num~=0     %If there are delay frames
                for k = 1:delay_num
                    delay(k).frames = stk_mod(delay(k).frames,'delete',[1:stim_num]);
                end
            end
            progressbar(i/trials);     %update the progress
        end
        %store experiment specific variables
        parsed_stk(1).horz_stim_num = horz_stim_num;
        parsed_stk(1).vert_stim_num = vert_stim_num;
    case 'visual_recruitment_sparsity'
        %this function orders the data from the above experiment, because
        %sparsity already has its data in sequential order in terms
        %of sparsity of stimuli we are just going to sort the stacks in
        %terms of trials.  The final structure parsed_stk, will
        %contain: parsed_stk(trials).stim_stk(stim_num) and
        %parse_stk(trials).delay_stks(delays).stk(stim_num).  So, each
        %trial has its own sorted stim stack and delay stacks.
        progressbar         %create progress bar and start timer
        for i = 1:trials
            parsed_stk(i).stim_stk = stk(:,:,1:stim_num);
            if delay_num~=0     %If there are delay frames
                for k = 1:delay_num
                    parsed_stk(i).delay_stks(k).stk= delay(k).frames(:,:,1:stim_num);
                end
            end
            %delete the stored frames
            stk = stk_mod(stk,'delete',[1:stim_num]);
            if delay_num~=0     %If there are delay frames
                for k = 1:delay_num
                    delay(k).frames = stk_mod(delay(k).frames,'delete',[1:stim_num]);
                end
            end
            progressbar(i/trials);     %update the progress
        end
end

%finalize the data structure
parsed_stk(1).stim_frames_trial = stim_num;         %the number of stim frames per trial.
parsed_stk(1).stim_frames_total = stim_frames;      %the number of stim frames in the original image stack.

%display processed information
disp(['This was a ',exp_type,' experiment.']);
disp(['This experiment had: ',num2str(delay_num),' Delay frames per stimulus.']);
disp(['This experiment had: ',num2str(trials),' Trials.']);
disp(['This experiment had: ',num2str(total_frames),' Total Frames.']);
disp(['This experiment had: ',num2str(stim_frames),' Total Stimulus Frames.']);
disp(['This experiment had: ',num2str(stim_num),' Stimulus frames per Trial.']);
disp(['This experiment had: ',num2str(delay_frames),' Frames per Delay.']);